<?php
/**
 * @package     LicenZer
 * @subpackage  com_licenzer
 *
 * @copyright   Copyright (C) 2024 LicenZer. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

defined('_JEXEC') or die;

use Joomla\CMS\Factory;

/**
 * LicenZer API Helper
 */
class LicenzerApiHelper
{
    private $apiUrl;
    private $token;

    public function __construct($token = null)
    {
        $params = ComponentHelper::getParams('com_licenzer');
        $this->apiUrl = $params->get('api_url', 'https://licenzer.pro');
        $this->token = $token ?: $params->get('reseller_token', '');
    }

    /**
     * Make API request
     */
    private function request($endpoint, $params = array())
    {
        $params['token'] = $this->token;
        $url = rtrim($this->apiUrl, '/') . '/' . ltrim($endpoint, '/') . '?' . http_build_query($params);

        try {
            $http = \Joomla\CMS\Http\HttpFactory::getHttp();
            $response = $http->get($url, null, 30);

            if ($response->code != 200) {
                return array('status' => 'error', 'message' => 'HTTP Error: ' . $response->code);
            }

            $data = json_decode($response->body, true);
            return $data ?: array('status' => 'error', 'message' => 'Invalid response');
        } catch (Exception $e) {
            return array('status' => 'error', 'message' => $e->getMessage());
        }
    }

    /**
     * Register new license
     */
    public function registerLicense($ip, $month = 1, $softwareKey = null)
    {
        $params = array(
            'ip' => $ip,
            'month' => $month,
        );

        if ($softwareKey) {
            $params['key'] = $softwareKey;
        }

        return $this->request('/resellerapi/register', $params);
    }

    /**
     * Activate license
     */
    public function activateLicense($ip, $softwareKey)
    {
        return $this->request('/resellerapi/activate', array(
            'ip' => $ip,
            'key' => $softwareKey,
        ));
    }

    /**
     * Deactivate license
     */
    public function deactivateLicense($ip, $softwareKey)
    {
        return $this->request('/resellerapi/deactivate', array(
            'ip' => $ip,
            'key' => $softwareKey,
        ));
    }

    /**
     * Delete license
     */
    public function deleteLicense($ip, $softwareKey)
    {
        return $this->request('/resellerapi/delete', array(
            'ip' => $ip,
            'key' => $softwareKey,
        ));
    }

    /**
     * Change license IP
     */
    public function changeIp($ip, $newIp, $softwareKey)
    {
        return $this->request('/resellerapi/changeiplicense', array(
            'ip' => $ip,
            'ip_new' => $newIp,
            'key' => $softwareKey,
        ));
    }

    /**
     * Get license info
     */
    public function getLicenseInfo($ip, $softwareKey)
    {
        return $this->request('/resellerapi/licenseinfo', array(
            'ip' => $ip,
            'key' => $softwareKey,
        ));
    }

    /**
     * Get all licenses
     */
    public function getLicenses()
    {
        return $this->request('/resellerapi/getlist');
    }

    /**
     * Get discount licenses
     */
    public function getDiscountLicenses($softwareKey = null)
    {
        $params = array();
        if ($softwareKey) {
            $params['key'] = $softwareKey;
        }
        return $this->request('/resellerapi/getdiscountlicenses', $params);
    }

    /**
     * Get available softwares
     */
    public function getSoftwares()
    {
        return $this->request('/resellerapi/getsoftwares');
    }

    /**
     * Get reseller balance
     */
    public function getBalance()
    {
        return $this->request('/resellerapi/getblanace');
    }

    /**
     * Test connection
     */
    public function testConnection()
    {
        $result = $this->request('/resellerapi/getstatus');
        return isset($result['status']) && $result['status'] == 'success';
    }
}

